#include "pch.h"

#include "Mem.hpp"
#include "Common.hpp"
#include "CfgParse.hpp"
#include "Misc.hpp"
#include "Links.h"
#include "Areas.h"
#include "Msg.hpp"
#include "Tic.hpp"
#include "crc32.hpp"
#include "Temp.hpp"
#include "File.h"

#include "CfgLexems.hpp"
#include "WELexems.h"

#include "MsgAttrs.h"

#include "LinksManager.h"
#include "AreasManager.h"

#include "Adrenalin.h"


/**
 * Implementation of CAdrenalin::ProcessInboundTics()
 * @param	ticsPath	path to look for tics
 * @param	filesPath	path to look for files for tics
 */
void CAdrenalin::ProcessInboundTics(LPCTSTR ticsPath, LPCTSTR filesPath) {
	// first of all lets check temp BSO: it can contain previous failed tosses
	checkTempBSO();

    // if path is specified for processed TICs we need to calculate their names
    CTicNamer   processedTicsNamer;

	WIN32_FIND_DATA FindFileData;
	HANDLE          hFindFile;
	BOOL            fFileFound;
	TCHAR           szFindFileMask[MAX_PATH];
	TCHAR           szFullFileName[MAX_PATH];
	TCHAR			szDestFilePath[MAX_PATH];
	TCHAR           szDestFileName[MAX_PATH];
	TCHAR           szNewAreasFileName[MAX_PATH];
	int             nProcessedTicNumber;

	VTString	vsTics;

	CTic            tic;

	vector<PLink>		forwards;

	CFile		newFilesFile;
	CFile		newAreasFile;

	UINT            uiRealCRC32;

	int             nTicNameFrom, i;

	TCHAR	szLine[LINE_SIZE];
	TCHAR   szLogLine[LINE_SIZE];
	TCHAR   szTimeLine[LINE_SIZE];

	int     *AKASeen;

	bool    fReplaceDone;
	TCHAR   szReplaceLine[LINE_SIZE];

	ifstream        in;

	CCfgParser     *TrafficParser;

	int             nTraffic;
	bool            fNewFilesArrived;
	bool            fTossedForLinks; // if yes, we should create TossFlag

	vector<tstring>	bodyLines;	// body lines
	vector<tstring>	warnings;	// warning lines
	vector<tstring>	errors;		// error lines
	CMsg		warnMessage;	// warn message to sysop

	// prepare log
	g_vars.m_log << CLog::ResetSingle(new CLog::StdOut());

	// print header (tossing information)

    if (g_vars.m_fShowMustGoOn) {
		g_vars.m_log << CLog::ResetSingle(new CLog::StdErr());
    } else {
		g_vars.m_log << CLog::SetSingle(new CLog::FileOut());
    }

	g_vars.m_log << TEXT("Tossing TICs in directory ") 
		         << ticsPath
		         << TEXT(":\n") << CLog::FlushStd()
		         << CLog::Pop();

    // make all fecho directories and set areas traffic to zero
	{
		int	areasCount = CAreasManager::getInstance()->getAreasCount();
		for (int i = 0; i < areasCount; i++) {
			PArea	area = CAreasManager::getInstance()->getArea( i );
			MakeDirStructure( area->getWorkingPath().c_str() );
			area->setTraffic( 0 );
		}
	}

    // make tics directories
    MakeDirStructure( g_vars.m_szBadTicsPath );
    MakeDirStructure( g_vars.m_szNewTicsPath );
    if (!IsEmptyStr(g_vars.m_szProcessedTicsPath)) {
        MakeDirStructure( g_vars.m_szProcessedTicsPath );
        processedTicsNamer.readTICs( g_vars.m_szProcessedTicsPath );
    }

    // read areas traffic
    
    lstrcpy( szFullFileName, m_szStartupPath );
    lstrcat( szFullFileName, TEXT("traffic.dat") );

    *szTimeLine = TEXT('\0');
    TrafficParser = new CCfgParser( szFullFileName );
    if (!TrafficParser->Failed()) {
        // read date line
        TrafficParser->ReadLine();
        if (!TrafficParser->Failed()) {
            lstrcpy( szTimeLine, TrafficParser->String() );
        }
        // read data lines
        while (!TrafficParser->Eof()) {
            TrafficParser->ReadLine();
            if (TrafficParser->Failed())
                break;
            if (TrafficParser->Count() != 2)
                break;
			PArea	area = CAreasManager::getInstance()->getArea( (*TrafficParser)[0] );
            if (area == NULL)
                continue;
	        if (!StrToInt( (*TrafficParser)[1], nTraffic ))
		        break;
            area->setTraffic( nTraffic );
        }
    }
    delete TrafficParser;
    
    lstrcpy( szFindFileMask, ticsPath );
    lstrcat( szFindFileMask, TEXT("*.ti?") );

    nProcessedTicNumber = 0;

    hFindFile = FindFirstFile( szFindFileMask, &FindFileData );
    fFileFound = (hFindFile != INVALID_HANDLE_VALUE);
    while (fFileFound) {
        if (!(FindFileData.dwFileAttributes &
              (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_HIDDEN |
               FILE_ATTRIBUTE_OFFLINE | FILE_ATTRIBUTE_READONLY |
               FILE_ATTRIBUTE_TEMPORARY)))
        {
		    vsTics.push_back( ticsPath );
		    vsTics.back() += FindFileData.cFileName;
        }
        fFileFound = FindNextFile( hFindFile, &FindFileData );
    }
    if (hFindFile != INVALID_HANDLE_VALUE)
        FindClose( hFindFile );

    // allocate memory for seen AKA
    AKASeen = new int[g_vars.m_vAKAAddr.size() + 1];
    // open newfiles list
    ConcatStrings( szFullFileName, m_szStartupPath, TEXT("newfiles.txt") );
    newFilesFile.Open( szFullFileName, GENERIC_WRITE, OPEN_ALWAYS );
    newFilesFile.SeekToEnd();
    // prepare file name for new areas
    ConcatStrings( szNewAreasFileName, m_szStartupPath, TEXT("newareas.txt") );
    // do not update traffic.dat if no new files arrived
    fNewFilesArrived = false;
    fTossedForLinks  = false;
    // process TICs
    vector<tstring>::iterator	itTic = vsTics.begin();
    while (itTic != vsTics.end()) {
	    // get name of tic without path
	    {
		    LPCTSTR	szTicName = (*itTic).c_str();
			int	i;
		    for ( nTicNameFrom = -1, i = 0; szTicName[i] != _T('\0'); i++) {
		        if (szTicName[i] == _T('\\'))
			        nTicNameFrom = i;
		    }
		    nTicNameFrom++;
	    }
        // trying to parse TIC
        if (tic.Read( (*itTic).c_str() )) {
            do {
                // check link

                PLink	link = CLinksManager::getInstance()->getLink( tic.m_addrFrom );
                if (link == NULL) {
			        lstrcpy( szLine, TEXT("    unknown sender (") );
			        tic.m_addrFrom.ToString( szLine + lstrlen(szLine) );
			        lstrcat( szLine, TEXT(")") );
			        // warn sysop if have to
			        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_ERRORS) {
			            errors.push_back( szLine );
			        }
			        _stprintf( szLogLine, TEXT("! %12s: %s"),
				               tic.getName(), szLine + 4 );
			        g_vars.m_log << szLogLine << TEXT_EOL;
					if (!tossingBadTics)
						MoveTicToBads( tic.getFullName(), nTicNameFrom );
			        break;
                }

                // check link activity  
                if (link->getActivity() == ACTIVITY_UNAVAILABLE) {
			        lstrcpy( szLine, TEXT("    unavailable sender (") );
			        tic.m_addrFrom.ToString( szLogLine + lstrlen(szLogLine) );
			        lstrcat( szLine, TEXT(")") );
			        // warn sysop if have to
			        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_ERRORS) {
			            errors.push_back( szLine );
			        }
			        _stprintf( szLogLine, TEXT("! %12s: %s"),
				               tic.getName(),
				               szLine + 4 );
			        g_vars.m_log << szLogLine << TEXT_EOL;
					if (!tossingBadTics)
						MoveTicToBads( tic.getFullName(), nTicNameFrom );
			        break;
                }

                // check password
                if (lstrcmpi( link->getPassword().c_str(), tic.m_szPW ) != 0) {
			        _stprintf( szLogLine,
			                   TEXT("! %12s: password error (%s instead of %s)"),
			                   tic.getName(), tic.m_szPW,
			                   link->getPassword().c_str() );
			        // warn sysop if have to
			        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_ERRORS) {
			            errors.push_back( szLogLine );
			        }
			        g_vars.m_log << szLogLine << TEXT_EOL;
					if (!tossingBadTics)
						MoveTicToBads( tic.getFullName(), nTicNameFrom );
			        break;
                }

                // form full file name of inbound file
                lstrcpy( szFullFileName, filesPath );
                lstrcat( szFullFileName, tic.getFileName() );

                // check file
                hFindFile = FindFirstFile( szFullFileName, &FindFileData );
                fFileFound = (hFindFile != INVALID_HANDLE_VALUE);
                if (!fFileFound) {
			        _stprintf( szLine, TEXT("    file %s not found"),
				               tic.getFileName() );
			        // warn sysop if have to
			        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_WARNINGS) {
			            warnings.push_back( szLine );
			        }
		            _stprintf( szLogLine, TEXT("! %12s: %s"),
                               tic.getName(), szLine + 4 );
			        g_vars.m_log << szLogLine << TEXT_EOL;
					if ((g_vars.maxTicAge >= 0) && (!tossingBadTics)) {
						if (isTicOld( tic.getFullName(), g_vars.maxTicAge )) {
							MoveTicToBads( tic.getFullName(), nTicNameFrom );
							lstrcpy( szLine, TEXT("    TIC was too old and moved into bad tics") );
							// warn sysop if have to
							if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_WARNINGS) {
								warnings.push_back( szLine );
							}
							_stprintf( szLogLine, TEXT("! %12s: %s"),
									   tic.getName(), szLine + 4 );
							g_vars.m_log << szLogLine << TEXT_EOL;
						}
					}
			        break;
                }
                FindClose( hFindFile );

                // check file finished or not
                if (FindFileData.dwFileAttributes &
                    (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_HIDDEN |
                     FILE_ATTRIBUTE_OFFLINE   | FILE_ATTRIBUTE_READONLY |
                     FILE_ATTRIBUTE_TEMPORARY))
                {
		            _stprintf( szLine, TEXT("    file %s is not accessible"),
	                           tic.getFileName() );
			        // warn sysop if have to
			        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_WARNINGS) {
			            warnings.push_back( szLine );
			        }
		            _stprintf( szLogLine, TEXT("! %12s: %s"),
	                           tic.getName(), szLine + 4 );
			        g_vars.m_log << szLogLine << TEXT_EOL;
			        break;
                }

                // check size
                int	nFileSize = ((FindFileData.nFileSizeHigh * MAXDWORD) +
				                 FindFileData.nFileSizeLow);

				// remember size
				filesize_t	fileSize;
				fileSize.LowPart  = FindFileData.nFileSizeLow;
				fileSize.HighPart = FindFileData.nFileSizeHigh;

		        if (tic.getSize() != -1) {
			        if (tic.getSize() != nFileSize) {
				        _stprintf( szLine,
				                   TEXT("    file size error (%ld instead of %ld)"),				     
				                   (FindFileData.nFileSizeHigh * MAXDWORD) +
				                   FindFileData.nFileSizeLow, tic.getSize() );
				        // warn sysop if have to
				        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_ERRORS) {
				            errors.push_back( szLine );
				        }
				        _stprintf( szLogLine,
				             TEXT("! %12s: %s"),
				             tic.getName(),
				             szLine + 4 );
				        g_vars.m_log << szLogLine << TEXT_EOL;
				        break;
			        }
                } else {
			        tic.setSize( nFileSize );
                }

                // check crc                
                if (!CalcCRC32( szFullFileName, uiRealCRC32 )) {
	                _stprintf( szLine,
                               TEXT("    failed to calculate CRC32 of file %s"),
                               tic.getFileName() );
			        // warn sysop if have to
			        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_ERRORS) {
			            errors.push_back( szLine );
			        }
	                _stprintf( szLogLine,
                               TEXT("! %12s: %s"),
                               tic.getName(), szLine + 4 );
		            g_vars.m_log << szLogLine << TEXT_EOL;
			        break;
                }

		        if (g_warnings.m_flagIncorrectCRC32) {
			        if (tic.isCRC32Present()) {
				        if (uiRealCRC32 != tic.m_uiCRC32) {
					        _stprintf( szLine,
				                       _T("    CRC32 warning (%08lX instead of %08lX)"),
					                   uiRealCRC32,
					                   tic.getCRC32() );
					        // warn sysop if have to
					        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_WARNINGS) {
						        warnings.push_back( szLine );
					        }
					        _stprintf( szLogLine,
					                   _T("! %12s: %s"),
					                   tic.getName(), szLine + 4 );
					        // turn on just FileOut
					        g_vars.m_log << CLog::SetSingle(new CLog::FileOut());
					        if (g_vars.m_fShowMustGoOn) {
					            g_vars.m_log << CLog::StdOut(true);
					        }
					        g_vars.m_log << szLogLine << TEXT_EOL;
					        // restore log output
					        g_vars.m_log << CLog::Pop();
					        // set correct CRC32
					        tic.setCRC32( uiRealCRC32 );
				        }
                    } else {
				        // tic hasn't crc32
				        lstrcpy( szLine, TEXT("    CRC32 warning (tag absent)") );
				        // warn sysop if have to
				        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_WARNINGS) {
					        warnings.push_back( szLine );
				        }
				        _stprintf( szLogLine, TEXT("! %12s: %s"),
					               tic.getName(), szLine + 4 );
				        // turn on just FileOut
				        g_vars.m_log << CLog::SetSingle(new CLog::FileOut());
				        if (g_vars.m_fShowMustGoOn) {
				            g_vars.m_log << CLog::StdOut(true);
				        }
				        g_vars.m_log << szLogLine << TEXT_EOL;
				        // restore log output
				        g_vars.m_log << CLog::Pop();
				        // set correct CRC32
				        tic.setCRC32( uiRealCRC32 );
				        tic.setCRC32Present( true );
			        }
                } else {
			        if (!tic.isCRC32Present()) {
				        lstrcpy( szLine, TEXT("    CRC32 error (tag absent)") );
				        // warn sysop if have to
				        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_ERRORS) {
					        errors.push_back( szLine );
				        }
				        _stprintf( szLogLine, TEXT("! %12s: %s"),
					               tic.getName(), szLine + 4 );
				        g_vars.m_log << szLogLine << TEXT_EOL;
				        break;
			        }
			        if (uiRealCRC32 != tic.getCRC32()) {
				        _stprintf( szLine,
				                   TEXT("    CRC32 error (%08lX instead of %08lX)"),
				                   uiRealCRC32, tic.getCRC32() );
				        // warn sysop if have to
				        if (g_vars.m_nWarnSysopByMail & WARN_SYSOP_ABOUT_ERRORS) {
					        errors.push_back( szLine );
				        }
				        _stprintf( szLogLine, TEXT("! %12s: %s"),
				                   tic.getName(), szLine + 4 );
				        g_vars.m_log << szLogLine << TEXT_EOL;
				        break;
			        }
		        }

		        // TODO: embed "warn sysop" code in code below

                // check area
				PArea	area = CAreasManager::getInstance()->getArea( tic.getAreaName() );
                if (area != NULL) {
					ConstTiePtr	tie = area->getTie( link );
                    if (tie == NULL) {
                        _stprintf( szLogLine, _T("! %12s: "),
				                   tic.getName() );
			            tic.m_addrFrom.ToString( szLogLine + lstrlen(szLogLine) );
			            _stprintf( szLogLine+lstrlen(szLogLine), _T(" isn't linked to %s\r\n"),
				                   tic.getAreaName() );
                        g_vars.m_log << szLogLine;
						if (!tossingBadTics)
							MoveTicToBads( tic.getFullName(), nTicNameFrom );
                        break;
                    }
                    if (!area->canSend( link )) {
                        _stprintf( szLogLine, _T("! %12s: "), tic.getName() );
			            tic.m_addrFrom.ToString( szLogLine+lstrlen(szLogLine) );
			            _stprintf( szLogLine+lstrlen(szLogLine), _T(" can't send into %s\r\n"),
                                   area->getName().c_str() );
                        g_vars.m_log << szLogLine;
						if (!tossingBadTics)
							MoveTicToBads( tic.getFullName(), nTicNameFrom );
                        break;
                    }
                } else {
                    if (link->canCreateAreas()) {
                        if (!CArea::isValidName( tic.getAreaName() )) {
                            _stprintf( szLogLine, _T("! %12s: area name %s is wrong\n"),
                                       tic.getName(),
                                       tic.getAreaName() );
                            g_vars.m_log << szLogLine;
							if (!tossingBadTics)
								MoveTicToBads( tic.getFullName(), nTicNameFrom );
                            break;
                        }
                        area = CAreasManager::getInstance()->autoCreate( tic.getAreaName(), link );

						ConstTiePtr	tie = area->getTie( link );

			            // log about new area

                        _stprintf( szLogLine, _T("+ area %s created by "), 
                                   area->getName().c_str() );
			            tic.m_addrFrom.ToString( szLogLine+lstrlen(szLogLine) );
			            lstrcat( szLogLine, _T("\r\n") );
			            // set log output only to FileOut
			            g_vars.m_log << CLog::SetSingle(new CLog::FileOut());
			            if (g_vars.m_fShowMustGoOn) {
				            // set log output to StdOut too
				            g_vars.m_log << CLog::StdOut(true);
			            }
                        g_vars.m_log << szLogLine;
			            // restore log output
			            g_vars.m_log << CLog::Pop();

                        newAreasFile.Open( szNewAreasFileName, GENERIC_WRITE, OPEN_ALWAYS );
			            newAreasFile.SeekToEnd();
                        _stprintf( szLine, _T("Fileecho area %s created by "), 
							area->getName().c_str() );
			            tic.m_addrFrom.ToString( szLine + lstrlen(szLine) );
			            lstrcat( szLine, _T("\r\n") );
			            newAreasFile.WriteStringAsOem( szLine );

			            // 2. We should check for previously kept forwards
			            // 2.1. Read forwards, if have to
			            if (!m_fForwardsRead) {
				            LoadForwards();
			            }
			            int	nForwardersLinked = 0;
			            for (CForwardInfoVector::iterator iter = m_vectForwardInfo.begin(); iter != m_vectForwardInfo.end();) {
				            // 2.4.5. Compare area with just created
				            if (lstrcmpi( iter->m_szArea, tic.getAreaName() ) == 0) {
					            // 2.5. Check address
					            SFTNAddress	addr;
					            if (ParseFTNAddress( iter->m_szLink, addr )) {
						            // 2.5.1. Link area
									PLink	forwardLink = CLinksManager::getInstance()->getLink( addr );
						            if (forwardLink != NULL) {
										if (area->canSend( forwardLink ) || area->canReceive( forwardLink )) {
											TiePtr	newTie = new CTie();
											newTie->setLink( forwardLink );
											newTie->setRouteType( ROUTE_NONE );
											area->addTie( newTie );
											CAreasManager::getInstance()->affectAreas();

								            // 2.5.2. Count links
								            nForwardersLinked++;
								            if (nForwardersLinked) {
									            newAreasFile.WriteOemString( "\r\nRequestor linked:\r\n" );
								            }
								            // 2.5.3. Prepare address string
								            lstrcpy( szLine, _T("  ") );
								            forwardLink->getAddress().ToString( szLine+2 );

								            // 2.5.4. Add log line

            								// set log output only to FileOut
								            g_vars.m_log << CLog::SetSingle(new CLog::FileOut());
								            if (g_vars.m_fShowMustGoOn) {
									            // set log output to StdOut too
									            g_vars.m_log << CLog::StdOut(true);
								            }
								            g_vars.m_log << TEXT("  linked ") << szLine+2 << TEXT_EOL;
								            // restore log output
								            g_vars.m_log << CLog::Pop();

								            lstrcat( szLine, TEXT("\r\n") );
								            // 2.5.5. Add newareas.txt line
								            newAreasFile.WriteStringAsOem( szLine );
							            } else {
								            // FIX: we should send message about failure
							            }
							            // remove info from forwards
							            iter = m_vectForwardInfo.erase(iter);
							            m_fForwardsChanged = true;
							            continue;
						            } else {
							            addr.ToString( szLine );
							            g_vars.m_log << TEXT("Unknown link ") << szLine
								                     << TEXT(" from ") << c_szForwardsFileName << TEXT_EOL;
						            }
                                } else {
						            g_vars.m_log << TEXT("Failed to parse address ") << iter->m_szLink << _T(" from ") << c_szForwardsFileName << TEXT_EOL;
					            }
				            }
				            ++iter;
			            }
			            if (nForwardersLinked > 0) {
				            newAreasFile.WriteOemString( "\r\n\r\n" );
			            }

                        newAreasFile.Close();
                    } else {
                        _stprintf( szLogLine, TEXT("! %12s: unknown area %s\r\n"),
                                  tic.getName(),
                                  tic.getAreaName() );
                        g_vars.m_log << szLogLine;
						if (!tossingBadTics)
							MoveTicToBads( tic.getFullName(), nTicNameFrom );
                        break;
                    }
                }

                // handle replaces if needed
                fReplaceDone = false;
                if (g_vars.m_fHandleReplaces && *(tic.m_szReplaces) != _T('\0')) {
                    lstrcpy( szDestFileName, area->getWorkingPath().c_str() );
                    lstrcat( szDestFileName, tic.m_szReplaces );
                    // check for file existance
                    if (_taccess( szDestFileName, 0 ) == 0) {
                        // remove file
                        if (_tunlink(szDestFileName) != 0) {
                            _stprintf( szReplaceLine,
                                       _T("? %12s: can't delete replaced file %s\r\n"),
                                       tic.getName(),
                                       szDestFileName );
                        } else {
                            area->clearDescFile( tic.m_szReplaces );
                            _stprintf( szReplaceLine,
                                       _T("                replaced file %s\r\n"),
                                       tic.m_szReplaces );
                        }
                        fReplaceDone = true;
                    }
                }
                // move file into fecho directory
		        lstrcpy( szDestFilePath, area->getWorkingPath().c_str() );
		        lstrcpy( szDestFileName, szDestFilePath );
                if (*tic.m_szFullName != _T('\0')) {
                    lstrcat( szDestFileName, tic.m_szFullName );
                } else {
                    lstrcat( szDestFileName, tic.getFileName() );
                }
                if (MyMoveFile( szFullFileName, szDestFileName ) != FO_SUCCEEDED) {
                    _stprintf( szLogLine,
                             _T("! %12s: failed to move file %s into file %s\r\n"),
                             tic.getName(), tic.getFileName(),
                             szDestFileName );
                    g_vars.m_log << szLogLine;
                    break;
                }
                area->increaseTraffic( (FindFileData.nFileSizeHigh * MAXDWORD) +
                                       FindFileData.nFileSizeLow );
                fNewFilesArrived = true;
                // output information in logfile and on screen, if needed
                _stprintf( szLogLine, _T("+ %12s: %-12s in %s\r\n"),
                           tic.getName(), tic.getFileName(),
                           area->getName().c_str() );

		        // set log output only to FileOut
		        g_vars.m_log << CLog::SetSingle(new CLog::FileOut());

        		if (g_vars.m_fShowMustGoOn) {
		        	// set log output to StdOut too
			        g_vars.m_log << CLog::StdOut(true);
		        }

                g_vars.m_log << szLogLine;

                if (fReplaceDone) {
                    g_vars.m_log << szReplaceLine;
                }

                // update descriptions
                if (*tic.m_szFullName != _T('\0')) {
                    g_vars.m_log << _T("                renamed to \"")
                                 << tic.m_szFullName << _T("\"\n");
                    area->updateDescFile( tic.m_szFullName, tic.m_szDesc, fileSize, 
                                          tic.m_listLDesc );
                } else {
                    area->updateDescFile( tic.getFileName(), tic.m_szDesc, fileSize, tic.m_listLDesc );
                }

		        // restore log output
		        g_vars.m_log << CLog::Pop();

                // tic is processed, lets treat the file
                TreatProcessedTic( tic.getFullName(), nTicNameFrom,
                                   processedTicsNamer );

                // update newfiles list

		        // Area
		        ConcatStrings( szLine, _T("Area "), area->getName().c_str() );
		        lstrcat( szLine, _T("\r\n") );
                newFilesFile.WriteStringAsOem( szLine );

		        // File
		        lstrcpy( szLine, _T("File ") );
                if (*tic.m_szFullName != _T('\0')) {
                    lstrcat( szLine, tic.m_szFullName );
                } else {
                    lstrcat( szLine, tic.getFileName() );
                }
		        lstrcat( szLine, _T("\r\n") );
		        newFilesFile.WriteStringAsOem( szLine );

		        // Size
		        _stprintf( szLine, _T("Size %lu\r\n"), 
			               (FindFileData.nFileSizeHigh * MAXDWORD) + FindFileData.nFileSizeLow );
		        newFilesFile.WriteStringAsOem( szLine );

		        // Desc
		        _stprintf( szLine, _T("Desc %s \r\n"), tic.m_szDesc );
		        newFilesFile.WriteStringAsOem( szLine );

		        // Origin
		        lstrcpy( szLine, _T("Origin ") );
		        tic.m_addrOrigin.ToStringWithoutDomain( szLine + lstrlen(szLine) );
		        lstrcat( szLine, _T("\r\n") );
		        newFilesFile.WriteStringAsOem( szLine );

		        // Uplink
		        lstrcpy( szLine, _T("Uplink ") );
		        link->getAddress().ToStringWithoutDomain( szLine + lstrlen(szLine) );
		        lstrcat( szLine, _T("\r\n") );
		        newFilesFile.WriteStringAsOem( szLine );

		        // keep file name in area's vector
                if (*tic.m_szFullName != _T('\0')) {
			        area->rememberNewFile( tic.m_szFullName );
                } else {
			        area->rememberNewFile( tic.m_szFile );
                }

                // add sender to seenby list, if absent
		        list<SFTNAddress>::iterator	iSeenby;
		        for (iSeenby = tic.m_listSeenby.begin(); iSeenby != tic.m_listSeenby.end(); iSeenby++) {
			        if (AddrCmp( link->getAddress(), (*iSeenby) ) == 0)
				        break;
		        }
                if (iSeenby == tic.m_listSeenby.end()) {
			        tic.m_listSeenby.push_back(link->getAddress());
                }
                // add us to seenby list, if absent
		        SFTNAddress	ourAddr = link->getOurAddress();
		        for (iSeenby = tic.m_listSeenby.begin(); iSeenby != tic.m_listSeenby.end(); iSeenby++) {			
			        if (AddrCmp( ourAddr, (*iSeenby) ) == 0)
				        break;
		        }
                if (iSeenby == tic.m_listSeenby.end()) {
			        tic.m_listSeenby.push_back( ourAddr );
                }
                // add absent AKAs if AllAKASeen switched to Yes
                if (g_vars.m_fAllAKASeen) {
                    for (i = 0; i <= g_vars.m_vAKAAddr.size(); i++) {
                        AKASeen[i] = 0;
                    }
		            for (iSeenby = tic.m_listSeenby.begin(); iSeenby != tic.m_listSeenby.end(); iSeenby++) {
                        // check main addr
                        if (AddrCmp( g_vars.m_addrMain, (*iSeenby) ) == 0) {
                            AKASeen[g_vars.m_vAKAAddr.size()] = 1;
                        }
                        // check AKAs
                        for (i = 0; i < g_vars.m_vAKAAddr.size(); i++) {
                            if (AddrCmp( g_vars.m_vAKAAddr[i], (*iSeenby) ) == 0) {
                                AKASeen[i] = 1;
                                break;
                            }
                        }
		            }
                    // add absent addresses
                    for (i = 0; i < g_vars.m_vAKAAddr.size(); i++) {
                        if (AKASeen[i] == 0) {
                            tic.m_listSeenby.push_back( g_vars.m_vAKAAddr[i] );
                        }
                    }
                    if (AKASeen[g_vars.m_vAKAAddr.size()] == 0) {
                        tic.m_listSeenby.push_back( g_vars.m_addrMain );
                    }
                }
                // do tics for all connected links absent in seenbys
                // first of all prepare list of links
				int	tiesCount = area->getTiesCount();
				int	i;
				for (i = 0; i < tiesCount; i++) {
					TiePtr	tie      = area->getTie( i );
					PLink	someLink = tie->getLink();
                    // skip passive or unavailable seenby
                    if (someLink->getActivity() != ACTIVITY_USUAL)
                        continue;
                    // skip sender
                    if (AddrCmp( someLink->getAddress(), link->getAddress()) == 0)
                        continue;
					if (!area->canReceive( someLink ))
						continue;
                    // check seenby
		            for (iSeenby = tic.m_listSeenby.begin(); iSeenby != tic.m_listSeenby.end(); iSeenby++) {
			            // debug
			            /*
			            TCHAR	szAddr[256];
			            g_vars.m_log << _T("Compare seenby and link: ");
			            (*itCurLinked).m_pLink->m_addr.ToString( szAddr );
			            g_vars.m_log << szAddr << " vs ";
			            (*iSeenby).ToString( szAddr );
			            g_vars.m_log << szAddr << _T("\n");
			            */
			            if (AddrCmp( someLink->getAddress(), (*iSeenby) ) == 0)
				            break;
		            }
                    if (iSeenby != tic.m_listSeenby.end())
                        continue;
                    // add link to list
                    // doesn't matter what is the value of RouteType
		            forwards.push_back( someLink );
                }
                // now do tics
                if (!forwards.empty())
                    fTossedForLinks = true;
				for (i = 0; i < forwards.size(); i++) {
                    tic.Forward( forwards[i], area, szDestFilePath, forwards );
                }
                // free forwards list
		        forwards.clear();

		        // create flags
		        {
                    if (*tic.m_szFullName != TEXT('\0')) {
			            lstrcpy( szLine, tic.m_szFullName );
                    } else {
			            lstrcpy( szLine, tic.m_szFile );
                    }

			        // 1) check area flags

			        int	count = area->getFileFlagsCount();

			        for (i = 0; i < count; i++) {
						CFileFlag&	flag = area->getFileFlag(i);
				        if (!flag.isUsed() &&
					        CheckMask( szLine, flag.getMask().c_str() ))
				        {
					        if (CreateFileFlag( flag.getFile().c_str()) ) {
						        flag.setUsed( true );

                                if (g_vars.m_fShowMustGoOn) {
							        g_vars.m_log << CLog::ResetSingle(new CLog::StdErr());
                                } else {
							        g_vars.m_log << CLog::SetSingle(new CLog::FileOut());
                                }

						        g_vars.m_log << _T("                file flag created: ")
							             << flag.getFile()
							             << TEXT_EOL
							             << CLog::Pop();
					        } else {
						        g_vars.m_log << _T("                failed to create file flag: ")
							                 << flag.getFile()
							                 << TEXT_EOL;
					        }
				        }
			        }

			        // 2) check common flags

    			    count = m_vsCommonFileFlags.size();

			        for (i = 0; i < count; i++) {
				        if (m_vbCommonFileFlagsUsed[i] == false &&
					        CheckMask( szLine, m_vsCommonFileMasks[i].c_str() ))
				        {
					        if (CreateFileFlag(m_vsCommonFileFlags[i].c_str())) {
						        m_vbCommonFileFlagsUsed[i] = true;
                                if (g_vars.m_fShowMustGoOn) {
							        g_vars.m_log << CLog::ResetSingle(new CLog::StdErr());
                                } else {
							        g_vars.m_log << CLog::SetSingle(new CLog::FileOut());
                                }
						        g_vars.m_log << _T("                file flag created: ")
							             << m_vsCommonFileFlags[i]
							             << TEXT_EOL
							             << CLog::Pop();
					        }
					        else {
						        g_vars.m_log << _T("                failed to create file flag: ")
							             << m_vsCommonFileFlags[i]
							             << TEXT_EOL;
					        }
				        }
			        }
		        }

                break;
            } while (true);

	        // check for warn data for sysop
	        if (!warnings.empty() || !errors.empty()) {
		        _stprintf( szLogLine, TEXT("Notification conserning %12s"),
			              tic.getName() );
		        bodyLines.push_back( TEXT("") );
		        bodyLines.push_back( szLogLine );
		        if (!warnings.empty()) {
			        bodyLines.push_back( TEXT("") );
			        bodyLines.push_back( TEXT("  Warnings:") );
			        bodyLines.push_back( TEXT("") );
			        AppendVector( bodyLines, warnings );
			        warnings.clear();
		        }
		        if (!errors.empty()) {
			        bodyLines.push_back( TEXT("") );
			        bodyLines.push_back( TEXT("  Errors:") );
			        bodyLines.push_back( TEXT("") );
			        AppendVector( bodyLines, errors );
			        errors.clear();
		        }
	        }
        } else {
            _stprintf( szLogLine, _T("! %12s: error parsering file\n"),
                     tic.getName() );
            g_vars.m_log << szLogLine
			             << TEXT("                ") << tic.GetLastError().c_str()
			             << TEXT_EOL;
        }

	    // flush stdout & stderr
	    g_vars.m_log << CLog::FlushStd();

	    itTic++;
    }
    newFilesFile.Close();
    // check if newfiles.txt has no data (zero size) and remove it if it hasn't
    ConcatStrings( szFullFileName, m_szStartupPath, _T("newfiles.txt") );
    hFindFile = FindFirstFile( szFullFileName, &FindFileData );
    fFileFound = (hFindFile != INVALID_HANDLE_VALUE);
    if (fFileFound) {
        FindClose( hFindFile );
        if (((FindFileData.nFileSizeHigh * MAXDWORD) +
             FindFileData.nFileSizeLow) == 0)
            _tunlink( szFullFileName);
    }
    // save traffic, if changed
    if (fNewFilesArrived) {
	    ConcatStrings( szFullFileName, m_szStartupPath, _T("traffic.dat") );
	    CFile	trafficFile;
        if (!trafficFile.Open( szFullFileName, GENERIC_WRITE, CREATE_ALWAYS )) {
		    g_vars.m_log << _T("! Error writing traffic base file ") 
			             << szFullFileName << TEXT_EOL;
        } else {
            if (*szTimeLine == _T('\0')) {
                time_t curTime = time(NULL);
                trafficFile.WriteOemString( ctime( &curTime ) );
            } else {
                trafficFile.WriteStringAsOem( szTimeLine );
		        trafficFile.WriteOemString( "\r\n" );
	        }

			int	areasCount = CAreasManager::getInstance()->getAreasCount();
			for (int i = 0; i < areasCount; i++) {
				PArea	area = CAreasManager::getInstance()->getArea( i );
			    if (area->getTraffic() > 0) {
				    _stprintf( szLine, 
					       _T("%s %lu\r\n"),
					       area->getName().c_str(),
					       area->getTraffic() );
				    trafficFile.WriteStringAsOem( szLine );
			    }
	        }
            trafficFile.Close();
        }
        // check if traffic.dat has no data (zero size) and remove it if it hasn't
        hFindFile = FindFirstFile( szFullFileName, &FindFileData );
        fFileFound = (hFindFile != INVALID_HANDLE_VALUE);
        if (fFileFound) {
            FindClose( hFindFile );
            if (((FindFileData.nFileSizeHigh * MAXDWORD) +
                 FindFileData.nFileSizeLow) == 0)
                _tunlink( szFullFileName );
        }
    }
    delete [] AKASeen;

    // warn sysop if have to
    if (!bodyLines.empty()) {
	    warnMessage.m_addrFrom = g_vars.m_addrMain;
	    warnMessage.m_addrTo   = g_vars.m_addrMain;
		CharToOem( g_vars.m_szSysOp, warnMessage.m_Header.m_szFromUserName );
	    strcpy( warnMessage.m_Header.m_szToUserName, "Adrenalin" );
	    warnMessage.Reply( ATTR_PRIVATE | ATTR_LOCAL, 0, "Notification", bodyLines );
    }

    // create flag, if tossed
    if (fTossedForLinks && *g_vars.m_szTossFlagName != _T('\0')) {
        CreateFileFlag( g_vars.m_szTossFlagName );
    }
    if (fNewFilesArrived) {
	    UpdateFilesDB();
    }

	// restore log
	g_vars.m_log << CLog::Pop();
}


void CAdrenalin::TreatProcessedTic(LPCTSTR fullTicName, int onlyNameFrom, 
                                   CTicNamer& processedTicsNamer)
{
    TCHAR   logLine[LINE_SIZE];

    if (IsEmptyStr(g_vars.m_szProcessedTicsPath)) {
        // no directory specified for processed tics
        // lets remove the tic
        if (_tunlink( fullTicName ) != 0) {
            wsprintf( logLine, _T("? %12s: can't delete TIC\n"),
                      fullTicName + onlyNameFrom );
            g_vars.m_log << logLine;
        }
    }
    else {
        // special directory specified for processed tics
        // lets try to move the tic there

	    // form tic name
        TCHAR   newTicName[MAX_PATH];
	    wsprintf( newTicName, TEXT("%sto%06lu.tic"), g_vars.m_szProcessedTicsPath,
                  processedTicsNamer.getFreeNumber() );
        if (MyMoveFile(fullTicName, newTicName)) {
            // TIC was moved
            wsprintf( logLine, _T("                after tossing TIC was moved to %s\n"),
                      newTicName + lstrlen(g_vars.m_szProcessedTicsPath) );
        }
        else {
            // We failed to move TIC
            wsprintf( logLine, _T("?               failed to move TIC to %s\n"),
                      newTicName + lstrlen(g_vars.m_szProcessedTicsPath) );
        }
        g_vars.m_log << logLine;
    }
}


bool CAdrenalin::processTic(CTic& tic, const tstring& filePath) {
	// check sender
	
	PLink	sender = CLinksManager::getInstance()->getLink( tic.getSenderAddress() );
	if (sender == NULL) {
		// TODO: log unknown sender
		return false;
	}

	// check sender activity

	if (sender->getActivity() == ACTIVITY_UNAVAILABLE) {
		// TODO: log inactive link
		return false;
	}

	// check password
	if (lstrcmpi( tic.getPassword(), sender->getPassword().c_str() ) != 0) {
		// TODO: log incorrect password
		return false;
	}

    // form full file name of inbound file
	tstring	fullFileName( filePath );
	fullFileName += tic.getFileName();

    // check file
	WIN32_FIND_DATA	fileData;
    HANDLE			hFile = FindFirstFile( fullFileName.c_str(), &fileData );
    if (hFile == INVALID_HANDLE_VALUE) {
		// TODO: log that tic wasn't found
		// TODO: move old tic to bad tics
		return false;
    }
    FindClose( hFile );

	// check file finished or not
	if (fileData.dwFileAttributes &
		(FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_HIDDEN |
		 FILE_ATTRIBUTE_OFFLINE   | FILE_ATTRIBUTE_READONLY |
		 FILE_ATTRIBUTE_TEMPORARY))
	{
		// TODO: log that file is not accessible
		return false;
	}

    // check size
    int	fileSize = ((fileData.nFileSizeHigh * MAXDWORD) + fileData.nFileSizeLow);

	if (tic.getSize() != -1) {
		if (tic.getSize() != fileSize) {
			// TODO: log that file size from tic differs from real file size
			return false;
		}
    } else {
		tic.setSize( fileSize );
    }

	// TODO: check CRC32

    // check area
	PArea	area = CAreasManager::getInstance()->getArea( tic.getAreaName() );
	TiePtr	tie  = NULL;
	if (area != NULL) {
		tie = area->getTie( sender );
        if (tie == NULL) {
			// TODO: log that sender is not linked to the area
            return false;
        }
		if (!area->canSend( sender )) {
			// TODO: log that sender cannot send into area
			return false;
		}
    } else {
		// we have no such area, let's check whether sender is allowed to create it
		if (!sender->canCreateAreas()) {
			// TODO: log unknown area
			return false;
		}
		if (CArea::isValidName(tic.getAreaName())) {
			// TODO: log that area name is invalid
            return false;
        }
		area = CAreasManager::getInstance()->autoCreate( tic.getAreaName(), sender );
		assert( area != NULL );
		tie  = area->getTie( sender );
		assert( tie != NULL );

		// TODO: log that new area created
		// TODO: write info about new area

		// TODO: check previously kept forwards
	}

	// TODO: handle replaces

	// TODO: move file into the fileecho directory

	// TODO: update descriptions

	return true;
}
